﻿using log4net;
using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using VA.PPMS.IWS.BlobService.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.OrchestrationService.Interface;
using VA.PPMS.IWS.QueueService.Interface;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.OrchestrationService
{
    public class OrchestrationService : IOrchestrationService
    {
        private readonly ILog _logger;
        private readonly IBlobService _blobService;
        private readonly IQueueService _queueService;

        public OrchestrationService(ILog logger, IBlobService blobService, IQueueService queueService)
        {
            _logger = logger;
            _blobService = blobService;
            _queueService = queueService;
        }

        public async Task StartProcessingAsync(DasMessage message, Providers providers)
        {
            _logger.Info($"@@@@ INFO - Start Main Orchestration Service @@@@");

            try
            {
                RunAsync(ProcessPayload(message, providers));

                await Task.Run(() => { });

                _logger.Info($"@@@@ INFO - End Main Orchestration Service @@@@");
            }
            catch (Exception ex)
            {
                _logger.Error($"@@@@ ERROR - There was a problem with the Main Orchestration Service.", ex);
                throw new PpmsServiceException($"There was a problem with the Main Orchestration Service.", ex);
            }
        }

        private async Task ProcessPayload(DasMessage message, Providers providers)
        {
            int i = 0;
            int j = 1;
            int chunkSize = 1000;
            DasMessage subMsg = null;
            string dasMsgInit = message.ToString();

            if (providers == null || providers.Provider == null)
                throw new PpmsServiceException($"Unable to process payload, the provider list is invalid.");

            // Initialize provider list
            Providers subList = InitProviders();

            foreach (var item in providers.Provider)
            {
                if (i != 0 && i % chunkSize == 0)
                {
                    subMsg = new DasMessage(dasMsgInit)
                    {
                        Index = j.ToString()
                    };

                    _logger.Info($"----> Adding item to queue {i}:{j}");
                    await AddProvidersToQueue(subMsg, subList);
                    // Reinitialize subList
                    subList = InitProviders();
                    j++;
                }

                // Create element in sub list
                subList.Provider.Add(item);

                i++;
            }

            // Save remainder
            subMsg = new DasMessage(dasMsgInit)
            {
                Index = DasMessage.FinalFileIndicator
            };
            _logger.Info($"----> Adding final item to queue {i}:F");
            await AddProvidersToQueue(subMsg, subList);
        }

        private async Task AddProvidersToQueue(DasMessage message, Providers providers)
        {
            var ableToSave = await _blobService.PutBlobAsync(message.ToString(), providers);

            if (!ableToSave)
            {
                _logger.Error($"@@@@ ERROR - Unable to store Provider Profile in Blob Storage @@@@");
                throw new PpmsServiceException($"Unable to store Provider data into Blob Storage");
            }

            await _queueService.PutDasQueueAsync(message);
        }

        private Providers InitProviders()
        {
            return new Providers()
            {
                Provider = new List<Provider>()
            };
        }

        private void RunAsync(Task task)
        {
            task.ContinueWith(t =>
            {
                _logger.Error("!!!! ERROR - Orchestration service error on asynchronous call !!!!", t.Exception);

            }, TaskContinuationOptions.OnlyOnFaulted);
        }
    }
}